#include <WiFi.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <Adafruit_SHT31.h>
#include <NTPClient.h>
#include <WiFiUdp.h>

// ===== WiFi info =====
// 硬编码 WiFi 凭证
const char* ssid     = ""; // 无线名称
const char* password = ""; // 无线密码

// ===== OLED setup =====
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 32
// 初始化 OLED：128x32 像素，I2C 接口
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, -1);

// ===== SHT31 setup =====
Adafruit_SHT31 sht31 = Adafruit_SHT31(); // 默认 I2C 地址 0x44

// ===== NTP setup =====
WiFiUDP ntpUDP;
NTPClient timeClient(ntpUDP, "pool.ntp.org", 8 * 3600, 60000); // UTC+8，更新间隔 60s

// ===== Pins  =====
#define I2C_SDA 8
#define I2C_SCL 9
#define LED_PIN 0 // ESP32 内置 LED

// LED
unsigned long lastBlink = 0;

void setup() {
  Serial.begin(115200); // 串口调试，波特率 115200
  Wire.begin(I2C_SDA, I2C_SCL); // 初始化 I2C，总线引脚自定义

  // WiFi 连接：阻塞式等待，打印点号进度
  WiFi.begin(ssid, password);
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("\nWiFi connected!"); // 成功后打印 IP（未显示，可添加 WiFi.localIP()）

  // OLED 初始化：切换到内部电压源，地址 0x3C；失败则死循环
  if (!display.begin(SSD1306_SWITCHCAPVCC, 0x3C)) {
    Serial.println(F("SSD1306 allocation failed"));
    for (;;); // 无限循环，需手动重启
  }
  display.clearDisplay();
  display.setRotation(1); // 旋转 90° 适配横屏
  display.display();

  // SHT31 初始化：地址 0x44，失败则死循环
  if (!sht31.begin(0x44)) {
    Serial.println("Couldn't find SHT31");
    while (1) delay(1);
  }

  // NTP 初始化：启动 UDP 客户端
  timeClient.begin();

  // LED 初始化：输出模式，低电平（灭）
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LOW);
}

void loop() {
  timeClient.update(); // 更新 NTP 时间（每 60s 同步一次）

  int hours = timeClient.getHours();
  int minutes = timeClient.getMinutes();
  int seconds = timeClient.getSeconds();

  // 转换为 12 小时制：处理 AM/PM，午夜为 12
  bool isPM = false;
  if (hours >= 12) {
    isPM = true;
    if (hours > 12) hours -= 12;
  }
  if (hours == 0) hours = 12;

  // 读取传感器：摄氏温度和相对湿度
  float tempC = sht31.readTemperature();
  float hum   = sht31.readHumidity();

  // 格式化时间字符串：两位数填充零
  char hStr[3], mStr[3], sStr[3];
  sprintf(hStr, "%02d", hours);
  sprintf(mStr, "%02d", minutes);
  sprintf(sStr, "%02d", seconds);

  // ===== OLED 显示逻辑 =====
  display.clearDisplay();// 清屏
  display.setTextColor(SSD1306_WHITE);// 白色文本

  // 表头：小字体 "/////"（装饰线？），AM/PM，水平分隔线
  display.setTextSize(0.5);
  display.setCursor(2, 0);
  display.println("/////");
  display.setCursor(60, 10);
  display.println(isPM ? "PM" : "AM");
  display.drawLine(0, 10, 128, 10, SSD1306_WHITE); // Y=10 横线

  // 时间显示：大字体 hh mm ss，但坐标需调整（屏幕高 32px，旋转后逻辑变化）
  // 小时
  display.setTextSize(2);
  display.setCursor(5, 30); // hh 位置（可能溢出，需验证）
  display.println(hStr);

  // 分
  display.setTextSize(2);
  display.setCursor(5, 52); // mm（Y=52 超出 32px？旋转后坐标系变）
  display.println(mStr);

  // 秒
  display.setTextSize(2);
  display.setCursor(5, 72);
  display.println(sStr);

  // 分隔线：Y=95（远超屏幕）
  display.drawLine(0, 95, 128, 95, SSD1306_WHITE);

  // 温湿度：小字体，整数显示，° 符号用 ASCII 247
  display.setTextSize(1);
  display.setCursor(5, 105); // 位置溢出
  display.print((int)tempC);
  display.print((char)247); // °
  display.print("C");
  display.setCursor(5, 120); // 溢出
  display.print((int)hum);
  display.print("%");

  display.display(); // 更新屏幕

  // ===== LED 心跳：每 5s 闪烁 20ms =====
  unsigned long now = millis(); // 非阻塞计时
  if (now - lastBlink >= 5000) {
    digitalWrite(LED_PIN, HIGH);
    delay(20); // 阻塞 20ms（小影响）
    digitalWrite(LED_PIN, LOW);
    lastBlink = now;
  }

  delay(200); // 循环延时，降低 CPU 负载（约 5Hz 更新率）
}
